<?php

namespace WP_Defender\Behavior\Scan;

use Calotes\Component\Behavior;
use WP_Defender\Behavior\WPMUDEV;
use WP_Defender\Component\Timer;
use WP_Defender\Model\Scan_Item;
use WP_Defender\Traits\IO;
use WP_Defender\Model\Setting\Scan as Scan_Settings;

class Malware_Scan extends Behavior {
	use IO;

	const YARA_RULES = 'defender_yara_rules';
	/**
	 * Cache for file content
	 * @var string
	 */
	private $content;

	/**
	 * Cache tokens
	 * @var
	 */
	private $tokens;

	/**
	 * Backup memory
	 * @var
	 */
	private $memory;

	/**
	 * @param object $scan_settings
	 *
	 * @return array
	 */
	protected function get_additional_rules( $scan_settings ) {
		$theme_cache          = false;
		$plugin_cache         = false;
		$theme_slugs_changes  = array();
		$plugin_slugs_changes = array();
		//Checked Theme option
		if ( $scan_settings->integrity_check && $scan_settings->check_themes ) {
			$theme_cache = true;
			$arr         = get_site_option( Theme_Integrity::THEME_SLUGS, false );
			if ( is_array( $arr ) && ! empty( $arr ) ) {
				$theme_slugs_changes = $arr;
			}
		}
		//Checked Plugin option
		if ( $scan_settings->integrity_check && $scan_settings->check_plugins ) {
			$plugin_cache = true;
			$arr          = get_site_option( Plugin_Integrity::PLUGIN_SLUGS, false );
			if ( is_array( $arr ) && ! empty( $arr ) ) {
				$plugin_slugs_changes = $arr;
			}
		}

		return array(
			'theme_change'         => $theme_cache,
			'plugin_change'        => $plugin_cache,
			//list of themes with modifications
			'theme_slugs_changes'  => $theme_slugs_changes,
			//list of plugins with modifications
			'plugin_slugs_changes' => $plugin_slugs_changes,
		);
	}

	/**
	 * @param string $file_path
	 * @param array $rules
	 *
	 * @return bool
	 */
	protected function was_modificated_file( $file_path, $rules ) {
		//Unchecked options of Theme && Plugin change file, so green light to display Suspicious checks
		if ( ! $rules['theme_change'] && ! $rules['plugin_change'] ) {
			return true;
		}

		$search_on_theme = WP_CONTENT_DIR . DIRECTORY_SEPARATOR . 'themes/';
		if ( $rules['theme_change'] && false !== stripos( $file_path, $search_on_theme ) ) {
			//Suspicious code in theme file
			if ( empty( $rules['theme_slugs_changes'] ) ) {
				//no modifications
				return false;
			}
			$rev_file  = str_replace( $search_on_theme, '', $file_path );
			$matches   = explode( '/', $rev_file );
			$base_slug = array_shift( $matches );
			if ( in_array( $base_slug, $rules['theme_slugs_changes'], true ) ) {
				//modifications in this theme
				return true;
			}
			//modifications not here
			return false;
		}

		$search_on_plugin = WP_PLUGIN_DIR . '/';
		if ( $rules['plugin_change'] && false !== stripos( $file_path, $search_on_plugin ) ) {
			//Suspicious code in plugin file
			/**
			 * Empty list of plugin slugs may be not plugin modifications
			 * or not plugins on site
			 */
			if ( empty( $rules['plugin_slugs_changes'] ) ) {
				//no modifications
				return false;
			}
			$rev_file  = str_replace( $search_on_plugin, '', $file_path );
			$matches   = explode( '/', $rev_file );
			$base_slug = array_shift( $matches );
			if ( in_array( $base_slug, $rules['plugin_slugs_changes'], true ) ) {
				//modifications in this plugin
				return true;
			}
			//modifications not here
			return false;
		}

		//Other WP places
		return true;
	}

	public function suspicious_check() {
		set_time_limit( 0 );
		$this->prepare_emergency_shutdown();
		$timer = new Timer();
		$rules = $this->fetch_yara_rules();
		$this->attach_behavior( Malware_Quick_Scan::class, Malware_Quick_Scan::class );
		$this->attach_behavior( Malware_Deep_Scan::class, Malware_Deep_Scan::class );
		$model = $this->owner->scan;
		$pos   = (int) $model->task_checkpoint;

		$files = get_site_option( Gather_Fact::CACHE_CONTENT, array() );
		if ( empty( $files ) ) {
			return true;
		}
		$combinations = $this->get_additional_rules( new Scan_Settings() );
		$files        = new \ArrayIterator( $files );
		$files->seek( $pos );
		while ( $files->valid() ) {
			if ( ! $timer->check() ) {
				$this->log( 'Rage quit', 'malware_scan' );
				$model->save();
				break;
			}
			if ( $model->is_issue_ignored( $files->current() ) ) {
				$this->log( sprintf( 'skip %s because of file is ignored', $files->current() ), 'malware_scan' );
				$files->next();
				continue;
			}

			list( $result, $qs_detail ) = $this->do_quick_scan( $files->current(), $rules );
			if ( $result ) {
				$this->log( sprintf( 'file %s suspicious', $files->current() ), 'malware_scan' );
				$result = $this->do_deep_scan( $files->current(), $rules, $qs_detail );
				/**
				 * Add new item if Suspicious code is found and:
				 * plugins/themes are premium,
				 * plugins/themes are on wp.org but the code doesn't match from the WP repo (there are differences in checksums),
				 * deactivated options of File change detection > Scan plugin OR theme file changes
				*/
				if ( is_array( $result ) && $this->was_modificated_file( $files->current(), $combinations ) ) {
					$result['file'] = $files->current();
					$model->add_item( Scan_Item::TYPE_SUSPICIOUS, $result );
				}
			}
			$files->next();
			$model->task_checkpoint = $files->key();
			$model->calculate_percent( $files->key() * 100 / $files->count(), 6 );
			if ( $files->key() % 100 === 0 ) {
				//we should update the model percent each 100 files so we have some progress ont he screen$pos * 100 / $core_files->count()
				$model->save();
			}
		}

		if ( ! $files->valid() ) {
			$last = \WP_Defender\Model\Scan::get_last();
			if ( is_object( $last ) ) {
				$ignored_issues = $last->get_issues( Scan_Item::TYPE_SUSPICIOUS, Scan_Item::STATUS_IGNORE );
				foreach ( $ignored_issues as $issue ) {
					$this->owner->scan->add_item(
						Scan_Item::TYPE_SUSPICIOUS,
						$issue->raw_data,
						Scan_Item::STATUS_IGNORE
					);
				}
			}
			$model->task_checkpoint = null;
		}

		$model->save();

		return ! $files->valid();
	}

	/**
	 * We will use this for a safe switch when memory out happen
	 */
	public function prepare_emergency_shutdown() {
		$this->memory = str_repeat( '*', 1024 * 1024 );

		register_shutdown_function(
			function () {
				$this->memory = null;
				$err          = error_get_last();
				if (
					( ! is_null( $err ) )
					&& ( ! in_array( $err['type'], array( E_NOTICE, E_WARNING ), true ) )
				) {
					error_log( var_export( $err, true ) );
					$this->log( 'Something wrong happen, saving and quit', 'scan' );
					//$this->owner->scan->status = \WP_Defender\Model\Scan::STATUS_ERROR;
					++$this->owner->scan->task_checkpoint;
					$this->owner->scan->save();
				}
			}
		);
	}

	/**
	 * Fetch yara rules from API
	 *
	 * @return array|mixed
	 */
	private function fetch_yara_rules() {
		$rules = get_site_option( self::YARA_RULES, false );
		if ( is_array( $rules ) ) {
			return $rules;
		}
		$this->attach_behavior( WPMUDEV::class, WPMUDEV::class );
		$rules = $this->make_wpmu_request( WPMUDEV::API_SCAN_SIGNATURE );
		if ( is_array( $rules ) ) {
			update_site_option( self::YARA_RULES, $rules );

			return $rules;
		}

		return array();
	}
}